#!/bin/bash
# Copyright (C) 2018, Raffaello Bonghi <raffaello@rnext.it>
# All rights reserved
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
# 1. Redistributions of source code must retain the above copyright 
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. Neither the name of the copyright holder nor the names of its 
#    contributors may be used to endorse or promote products derived 
#    from this software without specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND 
# CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, 
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
# FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
# OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
# WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
# OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
# EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

# NVIDIA Identify version 
# reference: 
# https://devtalk.nvidia.com/default/topic/1014424/jetson-tx2/identifying-tx1-and-tx2-at-runtime/
# https://devtalk.nvidia.com/default/topic/996988/jetson-tk1/chip-uid/post/5100481/#5100481

# VERSION 	4.0
# DATE		2021/01/27

FN_LOG="log_version.txt"
FN_BDI="/proc/device-tree/nvidia,pmu-boardid"
FN_BDI2="/proc/device-tree/nvidia,boardids"

get_borad()
{
	JETSON_BOARD=""
	if [ -f /sys/module/tegra_fuse/parameters/tegra_chip_id ]; then
		case $(cat /sys/module/tegra_fuse/parameters/tegra_chip_id) in
			64)	JETSON_BOARD="TK1" ;;
			33)	JETSON_BOARD="TX1" ;;
			24)	JETSON_BOARD="TX2" ;;
			25)	JETSON_BOARD="Xavier" ;;
			*)	JETSON_BOARD="UNKNOWN" ;;
		esac
		JETSON_DESCRIPTION="NVIDIA Jetson $JETSON_BOARD"
	fi
	export JETSON_BOARD
	echo ${JETSON_BOARD}
}

get_borad_v2()
{
	BID=""
	JETSON_BOARD=""
	
	if [ -f ${FN_BDI} ]; then
		BID=$(cat ${FN_BDI})
	elif [ -f ${FN_BDI2} ]; then
		BID=$(cat -v ${FN_BDI2})
	fi

	if [[ $BID == *"3668"* ]]; then 
		JETSON_BOARD="NX"
	elif [[ $BID == *"3448"* ]]; then
		JETSON_BOARD="NANO"
	elif [[ $BID == *"2888"* ]]; then
		JETSON_BOARD="XAVIER"
	elif [[ $BID == *"3310"* ]]; then
		JETSON_BOARD="TX2"
	elif [[ $BID == *"3489"* ]]; then
		JETSON_BOARD="TX2i"
	elif [[ $BID == *"2180"* ]]; then
		JETSON_BOARD="TX1"
	else
		JETSON_BOARD="UNKNOWN"
	fi

	if [[ ${JETSON_BOARD} = *"UNKNOWN"* ]]; then
		get_borad
	else
		JETSON_DESCRIPTION="NVIDIA Jetson $JETSON_BOARD"
		export JETSON_BOARD
		echo ${JETSON_BOARD}
	fi
}

# NVIDIA Jetson version
# reference https://devtalk.nvidia.com/default/topic/860092/jetson-tk1/how-do-i-know-what-version-of-l4t-my-jetson-tk1-is-running-/
get_l4t_version()
{
	JETSON_JETPACK=""
	if [ -f /etc/nv_tegra_release ]; then
		# L4T string
		JETSON_L4T_STRING=$(head -n 1 /etc/nv_tegra_release)

		# Load release and revision
		JETSON_L4T_RELEASE=$(echo $JETSON_L4T_STRING | cut -f 1 -d ',' | sed 's/\# R//g' | cut -d ' ' -f1)
		JETSON_L4T_REVISION=$(echo $JETSON_L4T_STRING | cut -f 2 -d ',' | sed 's/\ REVISION: //g' )
		# unset variable
		unset JETSON_L4T_STRING
		
		# Write Jetson description
		JETSON_L4T="$JETSON_L4T_RELEASE.$JETSON_L4T_REVISION"
	fi
		
	export JETSON_L4T
	echo ${JETSON_L4T}
}

get_jetpack_version()
{
	JETSON_JETPACK=""
	if [ -f /etc/nv_tegra_release ]; then
		# L4T string
		JETSON_L4T_STRING=$(head -n 1 /etc/nv_tegra_release)

		# Load release and revision
		JETSON_L4T_RELEASE=$(echo $JETSON_L4T_STRING | cut -f 1 -d ',' | sed 's/\# R//g' | cut -d ' ' -f1)
		JETSON_L4T_REVISION=$(echo $JETSON_L4T_STRING | cut -f 2 -d ',' | sed 's/\ REVISION: //g' )
		# unset variable
		unset JETSON_L4T_STRING
		
		# Write Jetson description
		JETSON_L4T="$JETSON_L4T_RELEASE.$JETSON_L4T_REVISION"
		
		# Write version of jetpack installed
		# https://developer.nvidia.com/embedded/jetpack-archive
		#if [ "$JETSON_BOARD" = "Xavier" ] ; then 
		if [ "$1" = "Xavier" ] ; then 
			case $JETSON_L4T in
				"32.6.1") JETSON_JETPACK="4.6" ;;
				"32.5.1") JETSON_JETPACK="4.5.1" ;;
				"32.5")   JETSON_JETPACK="4.5" ;;
				"32.4.4") JETSON_JETPACK="4.4.1" ;;
				"32.4.3") JETSON_JETPACK="4.4" ;;
				"32.4.2") JETSON_JETPACK="4.4_DP" ;;
				"32.3.1") JETSON_JETPACK="4.3" ;;
				"32.2.1") JETSON_JETPACK="4.2.2/3" ;;
				"32.2.0") JETSON_JETPACK="4.2.1" ;;
				"32.1.0") JETSON_JETPACK="4.2" ;;
				"31.1.0") JETSON_JETPACK="4.1.1" ;;
				"31.0.1") JETSON_JETPACK="4.0" ;;
				*)        JETSON_JETPACK="UNKNOWN" ;;
			esac  
		#elif [ "$JETSON_BOARD" = "TX2i" ] ; then 
		elif [ "$1" = "TX2i" ] ; then 
			case $JETSON_L4T in
				"32.6.1") JETSON_JETPACK="4.6" ;;
				"32.5.1") JETSON_JETPACK="4.5.1" ;;
				"32.5")   JETSON_JETPACK="4.5" ;;
				"32.4.4") JETSON_JETPACK="4.4.1" ;;
				"32.4.3") JETSON_JETPACK="4.4" ;;
				"32.4.2") JETSON_JETPACK="4.4_DP" ;;
				"32.3.1") JETSON_JETPACK="4.3" ;;
				"32.2.1") JETSON_JETPACK="4.2.2/3" ;;
				"32.2.0") JETSON_JETPACK="4.2.1" ;;
				"32.1.0") JETSON_JETPACK="4.2" ;;
				"28.2.1") JETSON_JETPACK="3.2.1" ;;
				"28.2")   JETSON_JETPACK="3.2" ;;
				*)		  JETSON_JETPACK="UNKNOWN" ;;
			esac        
		#elif [ "$JETSON_BOARD" = "TX2" ] ; then
		elif [ "$1" = "TX2" ] ; then
			case $JETSON_L4T in
				"32.6.1") JETSON_JETPACK="4.6" ;;
				"32.5.1") JETSON_JETPACK="4.5.1" ;;
				"32.5")   JETSON_JETPACK="4.5" ;;
				"32.4.4") JETSON_JETPACK="4.4.1" ;;
				"32.4.3") JETSON_JETPACK="4.4" ;;
				"32.4.2") JETSON_JETPACK="4.4_DP" ;;
				"32.3.1") JETSON_JETPACK="4.3" ;;
				"32.2.1") JETSON_JETPACK="4.2.2/3" ;;
				"32.2.0") JETSON_JETPACK="4.2.1" ;;
				"32.1.0") JETSON_JETPACK="4.2" ;;
				"28.2.1") JETSON_JETPACK="3.2.1" ;;
				"28.2")   JETSON_JETPACK="3.2" ;;
				"28.1")   JETSON_JETPACK="3.1" ;;
				"27.1")   JETSON_JETPACK="3.0" ;;
				*)        JETSON_JETPACK="UNKNOWN" ;;
			esac
		#elif [ "$JETSON_BOARD" = "TX1" ] ; then
		elif [ "$1" = "TX1" ] ; then
			case $JETSON_L4T in	  
				"32.6.1")   JETSON_JETPACK="4.6" ;;
				"32.5.1")   JETSON_JETPACK="4.5.1" ;;
				"32.5")     JETSON_JETPACK="4.5" ;;
				"32.4.4")   JETSON_JETPACK="4.4.1" ;;
				"32.4.3")   JETSON_JETPACK="4.4" ;;
				"32.4.2")   JETSON_JETPACK="4.4_DP" ;;
				"32.3.1")   JETSON_JETPACK="4.3" ;;  			
				"32.2.3")   JETSON_JETPACK="4.2.3" ;;  			
				"32.2.1")   JETSON_JETPACK="4.2.2" ;;  			
				"32.2")   	JETSON_JETPACK="4.2.1" ;;  			
				"28.3.2")   JETSON_JETPACK="3.3.1" ;;  			
				"28.3.1")   JETSON_JETPACK="3.3.1" ;;
				"28.3")   	JETSON_JETPACK="3.3" ;;
				"28.2")   	JETSON_JETPACK="3.2 or 3.2.1" ;;
				"28.1")   	JETSON_JETPACK="3.1" ;;
				"24.2.3") 	JETSON_JETPACK="3.0" ;;
				"24.2.2") 	JETSON_JETPACK="3.0" ;;
				"24.2.1") 	JETSON_JETPACK="3.0 or 2.3.1" ;;
				"24.2")   	JETSON_JETPACK="2.3" ;;
				"24.1")   	JETSON_JETPACK="2.2.1 or 2.2" ;;
				"23.2")   	JETSON_JETPACK="2.1" ;;
				"23.1")   	JETSON_JETPACK="2.0" ;;
				*)        	JETSON_JETPACK="UNKNOWN" ;;
			esac
		#elif [ "$JETSON_BOARD" = "TK1" ] ; then
		elif [ "$1" = "TK1" ] ; then
			case $JETSON_L4T in
				"21.5") JETSON_JETPACK="2.3.1 or 2.3" ;;
				"21.4") JETSON_JETPACK="2.2 or 2.1 or 2.0 or DP 1.2" ;;
				"21.3") JETSON_JETPACK="DP 1.1" ;;
				"21.2") JETSON_JETPACK="DP 1.0" ;;
				*)      JETSON_JETPACK="UNKNOWN" ;;
			esac
		else
			# Unknown board
			JETSON_JETPACK="UNKNOWN"
		fi
	fi
	
	export JETSON_JETPACK
	echo ${JETSON_JETPACK}
}

# Read CUDA version
get_cuda_version()
{
	JETSON_CUDA=""
	if [ -f /usr/local/cuda/version.txt ]; then
		JETSON_CUDA=$(cat /usr/local/cuda/version.txt | sed 's/\CUDA Version //g')
	else
		JETSON_CUDA="NOT INSTALLED"
	fi
	export JETSON_CUDA
	echo ${JETSON_CUDA}
}

# Read opencv version
get_opencv_version()
{
	JETSON_OPENCV=""
	pkg-config --exists opencv
	if [ $? == "0" ] ; then
		JETSON_OPENCV=$(pkg-config --modversion opencv)
	else
		JETSON_OPENCV="NOT INSTALLED"
	fi
	export JETSON_OPENCV
	echo ${JETSON_OPENCV}
}

# get opencv version 
get_opencv_version2()
{
	if [ -f /usr/bin/opencv_version ]; then
		JETSON_OPENCV=$(/usr/bin/opencv_version)
		export JETSON_OPENCV
		echo ${JETSON_OPENCV}
	else
		get_opencv_version
	fi
}

function log_echo()
{
	echo $@ 
	echo $@ >> ${FN_LOG}
}

#export JETSON_BOARD
#export JETSON_CUDA
#export JETSON_JETPACK
#export JETSON_L4T

# TODO Add enviroments variables:
# - UID -> https://devtalk.nvidia.com/default/topic/996988/jetson-tk1/chip-uid/post/5100481/#5100481
# - GCID, BOARD, EABI
# - cuDNN
# - TensorRT
# - Visionworks

function main()
{
	L4T=$(get_l4t_version)
	RBOARD=$(get_borad_v2)
	RJETPACK=$(get_jetpack_version $RBOARD)
	RCUDA=$(get_cuda_version)
	ROPENCV=$(get_opencv_version2)
	
	sudo rm -rf ${FN_LOG}
	log_echo "L4T=${L4T}"
	log_echo "Board=${RBOARD}"
	log_echo "Jetpack=${RJETPACK}"
	log_echo "CUDA=${RCUDA}"
	log_echo "Opencv=${ROPENCV}"


# 	get_borad
#	get_jetpack_version
#	get_cuda_version
#	get_opencv_version
	
}

main $@
